# -*- coding: utf-8 -*-
"""Untitled5.ipynb

Automatically generated by Colab.

Original file is located at
    https://colab.research.google.com/drive/1RfoWE89I_vgSwVVh_GgCjOiesrj7mM5X
"""

import pandas as pd
import numpy as np
from datetime import datetime, timedelta
import holidays
import os
from io import BytesIO
from IPython.display import display, FileLink
import base64

# Configurazione parametri
ANNO = 2024
ITALY_HOLIDAYS = holidays.IT(years=ANNO)

# Profili di carico tipici ARERA (valori percentuali orari)
PROFILI_DOMESTICI = {
    '1.5kW': {
        'feriale': [0.027558, 0.025269, 0.024273, 0.023846,	0.023977, 0.025504, 0.028743, 0.030972, 0.030962, 0.030888, 0.030625, 0.030376,
        0.030569, 0.029836, 0.029265, 0.029381, 0.030594, 0.034162, 0.037788, 0.040439, 0.040196, 0.038267, 0.034856, 0.031225],
        'sabato': [0.030228, 0.027209, 0.02572, 0.02497, 0.024879, 0.026057, 0.028406, 0.030426, 0.032839, 0.034193, 0.034344, 0.034751,
        0.035685, 0.035221, 0.034673, 0.034008, 0.034553, 0.037319, 0.041169, 0.044332, 0.044379, 0.042515, 0.03884, 0.035045],
        'domenica': [0.032038, 0.028908, 0.026863, 0.026444, 0.026097, 0.026956, 0.029044, 0.030868, 0.034376, 0.036446, 0.037128, 0.036931,
0.03738, 0.03599, 0.034551, 0.033086, 0.033151, 0.03556, 0.038085, 0.040636, 0.041017, 0.039757, 0.036412, 0.032554]
    },
    '3.0kW': {
        'feriale': [0.139598, 0.114666, 0.102237, 0.0967, 0.095787, 0.102229, 0.125881, 0.158845, 0.161334, 0.159308, 0.155873, 0.160469,
       0.172159, 0.174441, 0.17002, 0.164846, 0.164508, 0.180254, 0.214046, 0.260485, 0.267253, 0.256292, 0.222106, 0.17865],
        'sabato': [0.143851, 0.118796, 0.104904, 0.098183, 0.096113, 0.099905, 0.11147, 0.138072, 0.171024, 0.191284, 0.193492, 0.196872,
        0.210723, 0.212785, 0.207493, 0.195265, 0.189656, 0.198725, 0.225874, 0.264231, 0.255645, 0.233392, 0.205454, 0.1752852105],
        'domenica': [0.148734, 0.12462, 0.107537, 0.101175, 0.097643, 0.099442, 0.107177, 0.27111, 0.162835, 0.197128, 0.215292, 0.229294,
        0.237571, 0.217856, 0.205789, 0.195098, 0.190065, 0.19904, 0.225771, 0.262656, 0.263633, 0.248472, 0.216493, 0.1774212119]
    },
    '4.5kW': {
        'feriale': [0.341157, 0.298783, 0.27899, 0.270695, 0.271387, 0.293052, 0.354282, 0.14263, 0.389171, 0.35462, 0.324325, 0.31504,
        0.327997, 0.326908, 0.323903, 0.323746, 0.338149, 0.381267, 0.457393, 0.563537, 0.572276, 0.54804, 0.484468, 0.4086424544],
        'sabato': [0.345266, 0.302139, 0.280148, 0.269484, 0.267818, 0.282071, 0.314166, 0.350449, 0.37242, 0.369491, 0.347704, 0.339267,
        0.364046, 0.362715, 0.357696, 0.34665, 0.352081, 0.384578, 0.444896, 0.526177, 0.518149, 0.478887, 0.432587, 0.386351],
        'domenica': [0.344672, 0.304295, 0.274133, 0.266473, 0.263631, 0.275394, 0.299931, 0.322485, 0.350163, 0.379229, 0.390401, 0.397078,
        0.41815, 0.387555, 0.366831, 0.360558, 0.367946, 0.401919, 0.463658, 0.548027, 0.551227, 0.521639, 0.46283, 0.395025]
    }
}

PROFILI_CONDOMINIALI = {
    '6.0kW': {
        'feriale': [0.758354, 0.706953, 0.682615, 0.673122, 0.684512, 0.745123, 0.858238, 0.929412, 0.874837, 0.778928, 0.69952, 0.661454,
        0.641522, 0.62475, 0.632276, 0.6513, 0.706778, 0.805518, 0.923382, 1.057, 1.0724, 1.0339, 0.9361, 0.8400],
        'sabato': [0.764439, 0.714597, 0.689772, 0.677455, 0.687537, 0.730955, 0.81065, 0.849291, 0.8, 9728, 0.731437, 0.658367, 0.623151,
        0.616599, 0.600145, 0.603326, 0.610445, 0.664124, 0.771293, 0.898118, 1.01336, 1.011358, 0.963168, 0.882921, 0.807106],
        'domenica': [0.758354, 0.706953, 0.682615, 0.673122, 0.684512, 0.745123, 0.858238, 0.929412, 0.874837, 0.778928, 0.69952, 0.661454,
        0.641522, 0.62475, 0.632276, 0.6513, 0.706778, 0.805518, 0.923382, 1.057162, 1.072378,1.03394, 0.936118, 0.839957]
    }
}

def is_festivo(data):
    """Controlla se la data è un giorno festivo"""
    return data in ITALY_HOLIDAYS or data.weekday() == 6

def is_sabato(data):
    """Controlla se la data è un sabato"""
    return data.weekday() == 5 and not is_festivo(data)

def get_tipo_giorno(data):
    """Restituisce il tipo di giorno"""
    if is_festivo(data):
        return 'domenica'
    elif is_sabato(data):
        return 'sabato'
    else:
        return 'feriale'

def calcola_fattore_normalizzazione(consumo_annuo, profili, anno):
    """Calcola il fattore di normalizzazione corretto"""
    profilo_base = []
    for data in pd.date_range(start=f'{anno}-01-01', end=f'{anno}-12-31', freq='D'):
        tipo_giorno = get_tipo_giorno(data)
        profilo_base.extend(profili[tipo_giorno])

    somma_profilo_base = sum(profilo_base)
    return consumo_annuo / somma_profilo_base

def crea_profilo_utente(consumo_annuo, profili, anno, tipo_utente='domestico'):
    """Crea il profilo orario per un utente"""
    date_range = pd.date_range(start=f'{anno}-01-01', end=f'{anno}-12-31 23:00', freq='H')
    consumi_orari = []

    # Calcola il fattore di normalizzazione
    fattore_normalizzazione = calcola_fattore_normalizzazione(consumo_annuo, profili, anno)

    # Genera i consumi orari
    for data in date_range:
        tipo_giorno = get_tipo_giorno(data)
        ora = data.hour

        # Consumo base
        consumo_base = profili[tipo_giorno][ora]

        # Variazione stagionale (diversa per domestico/condominiale)
        mese = data.month
        if tipo_utente == 'domestico':
            if mese in [12, 1, 2]:  # inverno
                fattore_stagionale = 1.2
            elif mese in [6, 7, 8]:  # estate
                fattore_stagionale = 0.85
            else:  # mezze stagioni
                fattore_stagionale = 1.0
        else:  # condominiale
            if mese in [12, 1, 2]:  # inverno (più illuminazione)
                fattore_stagionale = 1.1
            elif mese in [6, 7, 8]:  # estate
                fattore_stagionale = 0.95
            else:  # mezze stagioni
                fattore_stagionale = 1.0

        # Variazione casuale giornaliera (±15% domestico, ±10% condominiale)
        if tipo_utente == 'domestico':
            variazione_casuale = np.random.uniform(0.85, 1.15)
        else:
            variazione_casuale = np.random.uniform(0.9, 1.1)

        consumo_orario = consumo_base * fattore_normalizzazione * fattore_stagionale * variazione_casuale
        consumi_orari.append(consumo_orario)

    return pd.Series(consumi_orari, index=date_range, name='Consumo_kWh')

def create_download_link(filename):
    """Crea un link di download per il file"""
    with open(filename, 'rb') as f:
        data = f.read()
    b64 = base64.b64encode(data).decode()
    href = f'<a href="data:application/octet-stream;base64,{b64}" download="{filename}">Scarica {filename}</a>'
    return href

def main():
    """Funzione principale"""
    print("Generazione profili energetici...")

    # Utenti domestici
    utenze_domestiche = [
        {'potenza': '1.5kW', 'consumo_annuo': np.random.uniform(1500, 2000), 'nome': 'Domestico_1.5kW_1', 'tipo': 'domestico'},
        {'potenza': '3.0kW', 'consumo_annuo': 2700, 'nome': 'Domestico_3.0kW_1', 'tipo': 'domestico'},
        {'potenza': '3.0kW', 'consumo_annuo': 2700, 'nome': 'Domestico_3.0kW_2', 'tipo': 'domestico'},
        {'potenza': '3.0kW', 'consumo_annuo': 2700, 'nome': 'Domestico_3.0kW_3', 'tipo': 'domestico'},
        {'potenza': '3.0kW', 'consumo_annuo': 2700, 'nome': 'Domestico_3.0kW_4', 'tipo': 'domestico'},
        {'potenza': '3.0kW', 'consumo_annuo': 2700, 'nome': 'Domestico_3.0kW_5', 'tipo': 'domestico'},
        {'potenza': '3.0kW', 'consumo_annuo': 2700, 'nome': 'Domestico_3.0kW_6', 'tipo': 'domestico'},
        {'potenza': '3.0kW', 'consumo_annuo': 2700, 'nome': 'Domestico_3.0kW_7', 'tipo': 'domestico'},
        {'potenza': '3.0kW', 'consumo_annuo': 2700, 'nome': 'Domestico_3.0kW_8', 'tipo': 'domestico'},
        {'potenza': '3.0kW', 'consumo_annuo': 2700, 'nome': 'Domestico_3.0kW_9', 'tipo': 'domestico'},
        {'potenza': '3.0kW', 'consumo_annuo': 2700, 'nome': 'Domestico_3.0kW_10', 'tipo': 'domestico'},
        {'potenza': '4.5kW', 'consumo_annuo': np.random.uniform(2900, 3200), 'nome': 'Domestico_4.5kW_1', 'tipo': 'domestico'}
    ]

    # Utenti condominiali
    utenze_condominiali = [
        {'potenza': '6.0kW', 'consumo_annuo': 2269, 'nome': 'Condominiale_1', 'tipo': 'condominiale'},
        {'potenza': '6.0kW', 'consumo_annuo': 2586, 'nome': 'Condominiale_2', 'tipo': 'condominiale'}
    ]

    # Genera tutti i profili
    risultati = {}

    for utente in utenze_domestiche:
        print(f"Generando {utente['nome']}...")
        profilo = crea_profilo_utente(
            utente['consumo_annuo'],
            PROFILI_DOMESTICI[utente['potenza']],
            ANNO,
            utente['tipo']
        )
        risultati[utente['nome']] = profilo
        consumo_generato = profilo.sum()
        print(f"Consumo annuo generato: {consumo_generato:.1f} kWh (target: {utente['consumo_annuo']} kWh)")
        print(f"Deviazione: {(consumo_generato - utente['consumo_annuo'])/utente['consumo_annuo']*100:.1f}%")

    for utente in utenze_condominiali:
        print(f"Generando {utente['nome']}...")
        profilo = crea_profilo_utente(
            utente['consumo_annuo'],
            PROFILI_CONDOMINIALI[utente['potenza']],
            ANNO,
            utente['tipo']
        )
        risultati[utente['nome']] = profilo
        consumo_generato = profilo.sum()
        print(f"Consumo annuo generato: {consumo_generato:.1f} kWh (target: {utente['consumo_annuo']} kWh)")
        print(f"Deviazione: {(consumo_generato - utente['consumo_annuo'])/utente['consumo_annuo']*100:.1f}%")

    # Crea DataFrame completo
    df_completo = pd.DataFrame(risultati)

    # Salva i risultati
    filename_csv = 'profili_energetici_annuali.csv'
    filename_excel = 'profili_energetici_annuali.xlsx'

    df_completo.to_csv(filename_csv)
    df_completo.to_excel(filename_excel)

    # Statistiche
    print("\n" + "="*50)
    print("RIEPILOGO CONSUMI ANNUI:")
    print("="*50)
    for colonna in df_completo.columns:
        consumo_totale = df_completo[colonna].sum()
        print(f"{colonna}: {consumo_totale:.1f} kWh")

    print(f"\nFile salvati:")
    print(f"- {filename_csv}")
    print(f"- {filename_excel}")

    # Crea link di download automatico
    print("\n" + "="*50)
    print("DOWNLOAD AUTOMATICO:")
    print("="*50)

    # Per Jupyter Notebook
    try:
        from IPython.display import display, HTML
        display(HTML(create_download_link(filename_excel)))
        display(HTML(create_download_link(filename_csv)))
        print("I link di download sono stati generati sopra ↑")
    except:
        # Per ambiente non Jupyter
        print("Per scaricare i file, cercali nella cartella corrente:")
        print(f"- {os.path.abspath(filename_excel)}")
        print(f"- {os.path.abspath(filename_csv)}")

    return df_completo

if __name__ == "__main__":
    df = main()

    # Se eseguito in ambiente Jupyter, mostra anche un'anteprima
    try:
        from IPython.display import display
        print("\nAnteprima dei dati:")
        display(df.head())
        print("\nConsumi orari tipici (primi 5 valori):")
        for colonna in df.columns:
            print(f"{colonna}: {df[colonna].head(5).values} kWh")
    except:
        print("\nAnteprima dei dati:")
        print(df.head())
        print("\n...")
        print(df.tail())